// HappyPaintWork

// 選択された3つまたは4つのアンカーポイントを結んで白塗りの面を作成するスクリプト

(function() {
    if (app.documents.length === 0) {
        // ドキュメントが開かれていない場合は警告を出して中断
        alert("エラー: ドキュメントが開かれていません。");
        return;
    }

    var selection = app.activeDocument.selection;
    var selectedPoints = [];

    // 1. 選択されているアンカーポイントの座標を取得
    for (var i = 0; i < selection.length; i++) {
        var item = selection[i];
        if (item instanceof PathItem) {
            for (var j = 0; j < item.pathPoints.length; j++) {
                var point = item.pathPoints[j];
                if (point.selected === PathPointSelection.ANCHORPOINT) {
                    // 座標と、角度計算用のプロパティを格納
                    selectedPoints.push({
                        anchor: point.anchor,
                        x: point.anchor[0],
                        y: point.anchor[1],
                        angle: 0
                    });
                }
            }
        }
    }

    // 2. 選択されたアンカーポイント数の確認とエラー処理
    // 💡 3点または4点の場合に続行
    if (selectedPoints.length < 3 || selectedPoints.length > 4) {
        // 3点または4点選択されていない場合は警告を出して中断
        alert("エラー: 面を作成するには、ちょうど3つまたは4つのアンカーポイントをダイレクト選択ツールで選択する必要があります。\n現在選択されているアンカーポイント数: " + selectedPoints.length);
        return; 
    }
    
    // 選択された点の数を取得
    var pointCount = selectedPoints.length;

    // 3. 重心 (中心座標) を計算
    var sumX = 0;
    var sumY = 0;
    for (var k = 0; k < selectedPoints.length; k++) {
        sumX += selectedPoints[k].x;
        sumY += selectedPoints[k].y;
    }
    // 💡 選択された点の数で割って重心を計算
    var centerX = sumX / pointCount;
    var centerY = sumY / pointCount;
    
    // 4. 各点の中心からの角度を計算
    for (var l = 0; l < selectedPoints.length; l++) {
        var dx = selectedPoints[l].x - centerX;
        var dy = selectedPoints[l].y - centerY;
        selectedPoints[l].angle = Math.atan2(dy, dx);
    }
    
    // 5. 角度に基づいてソート（反時計回り）
    // (三角形でも四角形でも、この方法で点を並べることで交差しない面を作ります)
    selectedPoints.sort(function(a, b) {
        return a.angle - b.angle;
    });

    // 6. ソートされたオブジェクトから座標（anchor）のみを抽出 (mapの代替)
    var finalVertices = [];
    for (var m = 0; m < selectedPoints.length; m++) {
        finalVertices.push(selectedPoints[m].anchor);
    }

    // 7. 新しいパス（面）の作成
    var newPath = app.activeDocument.pathItems.add();
    newPath.closed = true;

    for (var n = 0; n < finalVertices.length; n++) {
        var newPoint = newPath.pathPoints.add();
        newPoint.anchor = finalVertices[n];
        newPoint.leftDirection = finalVertices[n];
        newPoint.rightDirection = finalVertices[n];
        newPoint.pointType = PointType.CORNER;
    }

    // 8. スタイルの設定
    newPath.filled = true;
    newPath.stroked = true;

    // ------------------------------------
    // 【修正点: 塗りをアクティブなデフォルトの塗りカラーに変更】
    // ------------------------------------
    try {
        // アクティブなドキュメントの「デフォルトの塗りカラー」を取得し適用
        // これは、ツールパネルの前面にある「塗り」の色を反映します
        newPath.fillColor = app.activeDocument.defaultFillColor;

    } catch(e) {
        // カラー設定でエラーが出た場合は塗りをオフ
        newPath.filled = false;
        // エラーメッセージのログ出力（ユーザーには見えない）
        // $.writeln("塗りカラー設定エラー: " + e); 
    }

    // 線のカラーはデフォルトを試みる（これはツールパネルの「線」の色を反映）
    newPath.strokeColor = app.activeDocument.defaultStrokeColor; 
    
    // 作成されたパスを選択状態にする
    app.activeDocument.selection = newPath;

})();