// グリッド状オブジェクトの間隔リアルタイム調整スクリプト
// スライダーまたは数値入力でリアルタイムに調整できます
// 横間隔と縦間隔の連動機能付き

#target illustrator

if (app.documents.length === 0) {
    alert("ドキュメントを開いてください。");
} else if (app.activeDocument.selection.length < 2) {
    alert("2つ以上のオブジェクトを選択してください。");
} else {
    main();
}

function main() {
    var doc = app.activeDocument;
    var sel = doc.selection;
    
    // 元の位置を保存
    var originalItems = [];
    for (var i = 0; i < sel.length; i++) {
        var bounds = sel[i].geometricBounds;
        originalItems.push({
            obj: sel[i],
            left: bounds[0],
            top: bounds[1],
            right: bounds[2],
            bottom: bounds[3],
            width: bounds[2] - bounds[0],
            height: bounds[1] - bounds[3]
        });
    }
    
    // 左上から順にソート
    originalItems.sort(function(a, b) {
        var yDiff = b.top - a.top;
        if (Math.abs(yDiff) > 5) {
            return yDiff;
        }
        return a.left - b.left;
    });
    
    // 列数を自動検出
    var columns = detectColumns(originalItems);
    
    // 初期間隔を計算
    var initialHSpace = 20;
    var initialVSpace = 20;
    if (originalItems.length >= 2) {
        if (columns > 1 && originalItems.length > columns) {
            initialHSpace = Math.round(originalItems[1].left - originalItems[0].right);
            initialVSpace = Math.round(originalItems[0].bottom - originalItems[columns].top);
        }
    }
    
    // ダイアログを作成
    var dialog = new Window("dialog", "グリッド間隔調整");
    dialog.alignChildren = "fill";
    
    // 連動チェックボックス
    var linkGroup = dialog.add("group");
    linkGroup.orientation = "row";
    linkGroup.alignment = "center";
    var linkCheckbox = linkGroup.add("checkbox", undefined, "横縦間隔を連動させる");
    linkCheckbox.value = false;
    
    // 横間隔
    var hGroup = dialog.add("group");
    hGroup.orientation = "row";
    hGroup.add("statictext", undefined, "横間隔:").preferredSize.width = 60;
    var hInput = hGroup.add("edittext", undefined, initialHSpace);
    hInput.preferredSize.width = 60;
    hGroup.add("statictext", undefined, "px");
    var hSlider = hGroup.add("slider", undefined, initialHSpace, -100, 500);
    hSlider.preferredSize.width = 200;
    
    // 縦間隔
    var vGroup = dialog.add("group");
    vGroup.orientation = "row";
    vGroup.add("statictext", undefined, "縦間隔:").preferredSize.width = 60;
    var vInput = vGroup.add("edittext", undefined, initialVSpace);
    vInput.preferredSize.width = 60;
    vGroup.add("statictext", undefined, "px");
    var vSlider = vGroup.add("slider", undefined, initialVSpace, -100, 500);
    vSlider.preferredSize.width = 200;
    
    // 列数
    var colGroup = dialog.add("group");
    colGroup.orientation = "row";
    colGroup.add("statictext", undefined, "列数:").preferredSize.width = 60;
    var colInput = colGroup.add("edittext", undefined, columns);
    colInput.preferredSize.width = 60;
    var colSlider = colGroup.add("slider", undefined, columns, 1, Math.min(originalItems.length, 20));
    colSlider.preferredSize.width = 200;
    
    // 情報表示
    dialog.add("statictext", undefined, "オブジェクト数: " + originalItems.length);
    
    // リセットボタン
    var resetBtn = dialog.add("button", undefined, "リセット");
    
    // ボタン
    var btnGroup = dialog.add("group");
    btnGroup.add("button", undefined, "OK", {name: "ok"});
    btnGroup.add("button", undefined, "キャンセル", {name: "cancel"});
    
    // 横間隔のスライダーイベント
    hSlider.onChanging = function() {
        var h = Math.round(this.value);
        hInput.text = h;
        
        // 連動がONの場合、縦間隔も更新
        if (linkCheckbox.value) {
            vInput.text = h;
            vSlider.value = h;
        }
        
        updateGrid();
    };
    
    // 横間隔の入力フィールドイベント
    hInput.onChanging = function() {
        var val = parseFloat(this.text);
        if (!isNaN(val)) {
            hSlider.value = Math.max(-100, Math.min(500, val));
            
            // 連動がONの場合、縦間隔も更新
            if (linkCheckbox.value) {
                vInput.text = val;
                vSlider.value = Math.max(-100, Math.min(500, val));
            }
            
            updateGrid();
        }
    };
    
    // 縦間隔のスライダーイベント
    vSlider.onChanging = function() {
        var v = Math.round(this.value);
        vInput.text = v;
        
        // 連動がONの場合、横間隔も更新
        if (linkCheckbox.value) {
            hInput.text = v;
            hSlider.value = v;
        }
        
        updateGrid();
    };
    
    // 縦間隔の入力フィールドイベント
    vInput.onChanging = function() {
        var val = parseFloat(this.text);
        if (!isNaN(val)) {
            vSlider.value = Math.max(-100, Math.min(500, val));
            
            // 連動がONの場合、横間隔も更新
            if (linkCheckbox.value) {
                hInput.text = val;
                hSlider.value = Math.max(-100, Math.min(500, val));
            }
            
            updateGrid();
        }
    };
    
    // 列数のスライダーイベント
    colSlider.onChanging = function() {
        var c = Math.round(this.value);
        colInput.text = c;
        updateGrid();
    };
    
    // 列数の入力フィールドイベント
    colInput.onChanging = function() {
        var val = parseInt(this.text);
        if (!isNaN(val) && val >= 1) {
            colSlider.value = Math.max(1, Math.min(Math.min(originalItems.length, 20), val));
            updateGrid();
        }
    };
    
    // グリッド更新関数
    function updateGrid() {
        var h = parseFloat(hInput.text);
        var v = parseFloat(vInput.text);
        var c = parseInt(colInput.text);
        
        if (!isNaN(h) && !isNaN(v) && !isNaN(c) && c >= 1) {
            applyGridSpacing(originalItems, h, v, c);
            app.redraw();
        }
    }
    
    // リセットボタン
    resetBtn.onClick = function() {
        restoreOriginalPositions(originalItems);
        hInput.text = initialHSpace;
        vInput.text = initialVSpace;
        colInput.text = columns;
        hSlider.value = initialHSpace;
        vSlider.value = initialVSpace;
        colSlider.value = columns;
        app.redraw();
    };
    
    // ダイアログを表示
    var result = dialog.show();
    
    // キャンセルの場合は元に戻す
    if (result != 1) {
        restoreOriginalPositions(originalItems);
        app.redraw();
    }
}

function applyGridSpacing(items, hSpace, vSpace, columns) {
    var startLeft = items[0].left;
    var startTop = items[0].top;
    
    for (var i = 0; i < items.length; i++) {
        var row = Math.floor(i / columns);
        var col = i % columns;
        
        var newLeft = startLeft + col * (items[0].width + hSpace);
        var newTop = startTop - row * (items[0].height + vSpace);
        
        var currentBounds = items[i].obj.geometricBounds;
        var deltaX = newLeft - currentBounds[0];
        var deltaY = newTop - currentBounds[1];
        
        items[i].obj.translate(deltaX, deltaY);
    }
}

function restoreOriginalPositions(items) {
    for (var i = 0; i < items.length; i++) {
        var currentBounds = items[i].obj.geometricBounds;
        var deltaX = items[i].left - currentBounds[0];
        var deltaY = items[i].top - currentBounds[1];
        items[i].obj.translate(deltaX, deltaY);
    }
}

function detectColumns(items) {
    if (items.length === 0) return 1;
    
    var firstTop = items[0].top;
    var cols = 1;
    
    for (var i = 1; i < items.length; i++) {
        if (Math.abs(items[i].top - firstTop) <= 5) {
            cols++;
        } else {
            break;
        }
    }
    
    return cols;
}