// ストックイラスト投稿前自動チェックスクリプト統合版
// Adobe Illustrator用
// 
// 使い方:
// このスクリプトを実行すると、自動的に以下の処理を実行します
// 1. ストックイラスト自動チェック
// 2. オープンパスをクローズパスに変換
// 
// ※ロックされているレイヤー、レイヤー名が「info」のレイヤーは処理対象外

(function() {
    if (app.documents.length === 0) {
        alert("ドキュメントが開かれていません。");
        return;
    }

    // 両方実行(自動チェック → クローズパス変換)
    runStockCheck();
    runCloseOpenPaths();
})();

// ========================================
// 1. ストックイラスト自動チェック
// ========================================
function runStockCheck() {
    var doc = app.activeDocument;
    var report = {
        closedPaths: 0,
        outlinedStrokes: 0,
        isolatedPoints: 0,
        outlinedTexts: 0,
        deletedImages: 0,
        deletedNoColorItems: 0,
        deletedEmptyLayers: 0
    };

    try {
        // 1. テキストのアウトライン化
        processTexts(doc, report);

        // 2. オープンパスの処理と孤立点の削除
        processPathItems(doc, report);

        // 3. ラスター画像の削除
        deleteRasterImages(doc, report);

        // 4. 色なしアイテムの削除
        deleteNoColorItems(doc, report);

        // 5. 空のレイヤーの削除
        deleteEmptyLayers(doc, report);

        // 処理結果レポート
        showStockCheckReport(report);

    } catch (e) {
        alert("エラーが発生しました: " + e.message);
    }
}

// テキストのアウトライン化
function processTexts(doc, report) {
    var textFrames = doc.textFrames;
    var count = 0;
    
    for (var i = textFrames.length - 1; i >= 0; i--) {
        try {
            // テキストが「info」レイヤーにある場合はスキップ
            if (textFrames[i].layer && textFrames[i].layer.name === "info") {
                continue;
            }
            
            textFrames[i].createOutline();
            count++;
        } catch (e) {
            // エラーが発生しても続行
        }
    }
    
    report.outlinedTexts = count;
}

// パスアイテムの処理
function processPathItems(doc, report) {
    var items = doc.pathItems;
    
    for (var i = items.length - 1; i >= 0; i--) {
        var item = items[i];
        
        try {
            // 「info」レイヤーのアイテムはスキップ
            if (item.layer && item.layer.name === "info") {
                continue;
            }
            
            // 孤立点のチェック(パスポイントが1つ以下)
            if (item.pathPoints.length <= 1) {
                item.remove();
                report.isolatedPoints++;
                continue;
            }

            // オープンパスの処理
            if (!item.closed) {
                // 塗りのみのオープンパス → パスを閉じる
                if (item.filled && !item.stroked) {
                    item.closed = true;
                    report.closedPaths++;
                }
                // 線のみのオープンパス → 線をアウトライン化
                else if (item.stroked) {
                    outlineStroke(item);
                    report.outlinedStrokes++;
                }
            }
        } catch (e) {
            // エラーが発生しても続行
        }
    }
}

// 線のアウトライン化
function outlineStroke(item) {
    try {
        item.selected = false;
        item.selected = true;
        app.executeMenuCommand("OffsetPath v22");
        item.selected = false;
    } catch (e) {
        try {
            if (item.stroked) {
                item.selected = true;
                app.executeMenuCommand("expandStyle");
                item.selected = false;
            }
        } catch (e2) {
            // 何もしない
        }
    }
}

// ラスター画像の削除
function deleteRasterImages(doc, report) {
    var count = 0;
    
    var rasterItems = doc.rasterItems;
    for (var i = rasterItems.length - 1; i >= 0; i--) {
        try {
            // 「info」レイヤーの画像はスキップ
            if (rasterItems[i].layer && rasterItems[i].layer.name === "info") {
                continue;
            }
            
            rasterItems[i].remove();
            count++;
        } catch (e) {}
    }
    
    var placedItems = doc.placedItems;
    for (var j = placedItems.length - 1; j >= 0; j--) {
        try {
            // 「info」レイヤーの画像はスキップ
            if (placedItems[j].layer && placedItems[j].layer.name === "info") {
                continue;
            }
            
            placedItems[j].remove();
            count++;
        } catch (e) {}
    }
    
    report.deletedImages = count;
}

// 色なしアイテムの削除
function deleteNoColorItems(doc, report) {
    var items = doc.pathItems;
    var count = 0;
    
    for (var i = items.length - 1; i >= 0; i--) {
        var item = items[i];
        
        try {
            // 「info」レイヤーのアイテムはスキップ
            if (item.layer && item.layer.name === "info") {
                continue;
            }
            
            if (!item.filled && !item.stroked) {
                item.remove();
                count++;
            }
        } catch (e) {}
    }
    
    report.deletedNoColorItems = count;
}

// 空のレイヤーの削除(再帰的)
function deleteEmptyLayers(doc, report) {
    var count = 0;
    
    function isLayerEmpty(layer) {
        if (layer.locked || !layer.visible) {
            return false;
        }
        
        // 「info」レイヤーは空とみなさない
        if (layer.name === "info") {
            return false;
        }
        
        if (layer.pageItems.length > 0) {
            return false;
        }
        
        if (layer.layers.length > 0) {
            for (var i = 0; i < layer.layers.length; i++) {
                if (!isLayerEmpty(layer.layers[i])) {
                    return false;
                }
            }
        }
        
        return true;
    }
    
    function removeEmptyLayers(parentLayer) {
        var layers = parentLayer ? parentLayer.layers : doc.layers;
        
        for (var i = layers.length - 1; i >= 0; i--) {
            var layer = layers[i];
            
            if (layer.layers.length > 0) {
                removeEmptyLayers(layer);
            }
            
            if (isLayerEmpty(layer)) {
                try {
                    layer.remove();
                    count++;
                } catch (e) {}
            }
        }
    }
    
    removeEmptyLayers(null);
    report.deletedEmptyLayers = count;
}

// ストックチェックレポート表示
function showStockCheckReport(report) {
    var message = "=== ストックイラスト自動チェック完了 ===\n\n";
    message += "パスを閉じた数: " + report.closedPaths + "\n";
    message += "アウトライン化した線: " + report.outlinedStrokes + "\n";
    message += "削除した孤立点: " + report.isolatedPoints + "\n";
    message += "アウトライン化したテキスト: " + report.outlinedTexts + "\n";
    message += "削除した画像: " + report.deletedImages + "\n";
    message += "削除した色なしアイテム: " + report.deletedNoColorItems + "\n";
    message += "削除した空レイヤー: " + report.deletedEmptyLayers + "\n\n";
    message += "※ロック中およびレイヤー名「info」は処理対象外\n\n";
    message += "処理が完了しました!";
    
    alert(message);
}

// ========================================
// 2. オープンパスをクローズパスに変換
// ========================================
function runCloseOpenPaths() {
    var doc = app.activeDocument;
    var processedCount = 0;

    // ドキュメント内のすべてのレイヤーを処理
    for (var i = 0; i < doc.layers.length; i++) {
        var layer = doc.layers[i];
        
        // ロックされていない、可視、かつレイヤー名が「info」でない場合のみ処理
        if (!layer.locked && layer.visible && layer.name !== "info") {
            processLayerForClosing(layer);
        }
    }
    
    // レイヤー内のすべてのアイテムを処理する関数
    function processLayerForClosing(layer) {
        // レイヤー内のパスアイテムを直接処理
        var pathItems = layer.pathItems;
        for (var j = 0; j < pathItems.length; j++) {
            var pathItem = pathItems[j];
            
            var hasFill = pathItem.filled;
            var hasNoStroke = !pathItem.stroked || pathItem.strokeWidth === 0;
            
            if (hasFill && hasNoStroke && !pathItem.closed) {
                pathItem.closed = true;
                processedCount++;
            }
        }
        
        // レイヤー内のグループを処理
        var groups = layer.groupItems;
        for (var k = 0; k < groups.length; k++) {
            processPathItemsForClosing(groups[k]);
        }
        
        // レイヤー内の複合パスを処理
        var compoundPaths = layer.compoundPathItems;
        for (var m = 0; m < compoundPaths.length; m++) {
            processPathItemsForClosing(compoundPaths[m]);
        }
    }

    // 再帰的にパスアイテムを処理する関数
    function processPathItemsForClosing(container) {
        var items = container.pathItems;
        
        for (var j = 0; j < items.length; j++) {
            var pathItem = items[j];
            
            var hasFill = pathItem.filled;
            var hasNoStroke = !pathItem.stroked || pathItem.strokeWidth === 0;
            
            if (hasFill && hasNoStroke && !pathItem.closed) {
                pathItem.closed = true;
                processedCount++;
            }
        }
        
        // グループ内のグループも処理
        if (container.typename === "GroupItem") {
            var groups = container.groupItems;
            for (var k = 0; k < groups.length; k++) {
                processPathItemsForClosing(groups[k]);
            }
        }
    }

    // 結果を表示
    var message = "=== オープンパス変換完了 ===\n\n";
    message += processedCount + "個のオープンパスをクローズパスに変換しました。\n\n";
    message += "※ロック中およびレイヤー名「info」は処理対象外";
    
    if (processedCount === 0) {
        message = "変換対象のオープンパス(面あり・線なし)が見つかりませんでした。\n\n";
        message += "※ロック中およびレイヤー名「info」は処理対象外";
    }
    
    alert(message);
}