// レイヤーマスクを1pxずつ調整するスクリプト
// 使い方: 
// - マスク付きレイヤーを選択して実行
// - ダイアログで「広げる」または「狭める」を選択
// - 適用後、履歴で戻して再度実行して調整可能

#target photoshop

function main() {
    if (app.documents.length === 0) {
        alert("ドキュメントが開かれていません。");
        return;
    }

    var doc = app.activeDocument;
    var layer = doc.activeLayer;

    // レイヤーマスクの存在確認
    if (!hasLayerMask(layer)) {
        alert("選択されたレイヤーにマスクがありません。");
        return;
    }

    // ダイアログを表示
    var dialog = new Window("dialog", "レイヤーマスク調整");
    dialog.alignChildren = "fill";

    // 情報表示
    var infoGroup = dialog.add("group");
    infoGroup.add("statictext", undefined, "現在のレイヤー: " + layer.name);

    // 調整量入力
    var pixelGroup = dialog.add("group");
    pixelGroup.add("statictext", undefined, "調整量 (px):");
    var pixelInput = pixelGroup.add("edittext", undefined, "0.5");
    pixelInput.characters = 5;
    
    // 矢印キーのイベントリスナー
    pixelInput.addEventListener("keydown", function(event) {
        var currentValue = parseFloat(this.text) || 0.5;
        
        // 上矢印キー
        if (event.keyName == "Up") {
            var newValue = currentValue + 0.5;
            this.text = newValue.toFixed(1);
            event.preventDefault();
        }
        // 下矢印キー
        else if (event.keyName == "Down") {
            if (currentValue > 0.5) {
                var newValue = currentValue - 0.5;
                this.text = newValue.toFixed(1);
            }
            event.preventDefault();
        }
    });

    // ラジオボタングループ
    var radioGroup = dialog.add("panel", undefined, "調整方向");
    radioGroup.alignChildren = "left";
    var expandRadio = radioGroup.add("radiobutton", undefined, "広げる (Expand)");
    var contractRadio = radioGroup.add("radiobutton", undefined, "狭める (Contract)");
    expandRadio.value = true;

    // 説明テキスト
    var helpGroup = dialog.add("group");
    helpGroup.alignment = "center";
    var helpText = helpGroup.add("statictext", undefined, 
        "※適用後、Ctrl+Z(履歴)で戻して\n再度実行することで微調整できます", 
        {multiline: true});
    helpText.characters = 40;

    // ボタングループ
    var buttonGroup = dialog.add("group");
    var okButton = buttonGroup.add("button", undefined, "適用", {name: "ok"});
    var cancelButton = buttonGroup.add("button", undefined, "キャンセル", {name: "cancel"});

    if (dialog.show() == 1) {
        var pixels = parseFloat(pixelInput.text);
        if (isNaN(pixels) || pixels < 0.5) {
            alert("有効なピクセル数を入力してください (最小: 0.5)");
            return;
        }

        var isExpand = expandRadio.value;
        adjustMask(doc, layer, pixels, isExpand);
    }
}

function hasLayerMask(layer) {
    try {
        var ref = new ActionReference();
        ref.putProperty(charIDToTypeID("Prpr"), charIDToTypeID("Usrs"));
        ref.putEnumerated(charIDToTypeID("Lyr "), charIDToTypeID("Ordn"), charIDToTypeID("Trgt"));
        var desc = executeActionGet(ref);
        return desc.hasKey(charIDToTypeID("Usrs"));
    } catch (e) {
        return false;
    }
}

function adjustMask(doc, layer, pixels, isExpand) {
    // マスクを選択
    var desc1 = new ActionDescriptor();
    var ref1 = new ActionReference();
    ref1.putEnumerated(charIDToTypeID("Chnl"), charIDToTypeID("Chnl"), charIDToTypeID("Msk "));
    desc1.putReference(charIDToTypeID("null"), ref1);
    desc1.putBoolean(charIDToTypeID("MkVs"), false);
    executeAction(charIDToTypeID("slct"), desc1, DialogModes.NO);

    // 全選択
    doc.selection.selectAll();

    if (isExpand) {
        // マスクを広げる (Maximum)
        doc.selection.expand(pixels);
    } else {
        // マスクを狭める (Minimum)
        doc.selection.contract(pixels);
    }

    // 選択範囲を反転してマスクに適用する方法
    // 実際にはMinimum/Maximumフィルタを使用
    if (isExpand) {
        // Maximum フィルタを適用
        var desc2 = new ActionDescriptor();
        desc2.putUnitDouble(charIDToTypeID("Rds "), charIDToTypeID("#Pxl"), pixels);
        executeAction(charIDToTypeID("Mxm "), desc2, DialogModes.NO);
    } else {
        // Minimum フィルタを適用
        var desc2 = new ActionDescriptor();
        desc2.putUnitDouble(charIDToTypeID("Rds "), charIDToTypeID("#Pxl"), pixels);
        executeAction(charIDToTypeID("Mnm "), desc2, DialogModes.NO);
    }

    // 選択解除
    doc.selection.deselect();

    // マスクを再選択（重要：レイヤー本体ではなくマスクを選択状態に保つ）
    var desc3 = new ActionDescriptor();
    var ref3 = new ActionReference();
    ref3.putEnumerated(charIDToTypeID("Chnl"), charIDToTypeID("Chnl"), charIDToTypeID("Msk "));
    desc3.putReference(charIDToTypeID("null"), ref3);
    desc3.putBoolean(charIDToTypeID("MkVs"), false);
    executeAction(charIDToTypeID("slct"), desc3, DialogModes.NO);
}

try {
    main();
} catch (e) {
    alert("エラーが発生しました: " + e.message + "\n行: " + e.line);
}