// Adobe Illustrator カレンダー作成スクリプト（アートボード版）
// 使い方: このスクリプトをIllustratorで実行してください

(function() {
    // ドキュメントが開いているかチェック
    if (app.documents.length === 0) {
        alert("先にドキュメントを開いてください。");
        return;
    }
    
    var doc = app.activeDocument;
    
    // 現在のアートボードのサイズを取得
    var currentArtboard = doc.artboards[doc.artboards.getActiveArtboardIndex()];
    var artboardRect = currentArtboard.artboardRect;
    var artboardWidth = artboardRect[2] - artboardRect[0];
    var artboardHeight = artboardRect[1] - artboardRect[3];
    
    // 設定ダイアログの表示
    var dialog = new Window("dialog", "カレンダー設定");
    
    // 年の入力
    dialog.add("statictext", undefined, "年:");
    var yearInput = dialog.add("edittext", undefined, new Date().getFullYear().toString());
    yearInput.characters = 6;
    
    // アートボードの配置
    dialog.add("statictext", undefined, "アートボードの配置:");
    var layoutGroup = dialog.add("group");
    var layoutGrid = layoutGroup.add("radiobutton", undefined, "5-5-2で並べる");
    var layoutHorizontal = layoutGroup.add("radiobutton", undefined, "横一列");
    layoutGrid.value = true;
    
    // カレンダーのサイズ
    dialog.add("statictext", undefined, "カレンダーサイズ (%):");
    var sizeGroup = dialog.add("group");
    var sizeInput = sizeGroup.add("edittext", undefined, "100");
    sizeInput.characters = 6;
    var resetBtn = sizeGroup.add("button", undefined, "リセット");
    resetBtn.onClick = function() {
        sizeInput.text = "100";
    };
    
    // 言語選択
    dialog.add("statictext", undefined, "言語:");
    var langGroup = dialog.add("group");
    var langJP = langGroup.add("radiobutton", undefined, "日本語");
    var langEN = langGroup.add("radiobutton", undefined, "英語");
    langJP.value = true;
    
    // 数字の位置
    dialog.add("statictext", undefined, "数字の位置:");
    var posGroup = dialog.add("group");
    var posCenter = posGroup.add("radiobutton", undefined, "センター");
    var posTopLeft = posGroup.add("radiobutton", undefined, "左上");
    posCenter.value = true;
    
    // グリッドの表示
    dialog.add("statictext", undefined, "グリッド:");
    var gridGroup = dialog.add("group");
    var gridYes = gridGroup.add("radiobutton", undefined, "あり");
    var gridNo = gridGroup.add("radiobutton", undefined, "なし");
    gridYes.value = true;
    
    // ボタン
    var btnGroup = dialog.add("group");
    btnGroup.add("button", undefined, "OK", {name: "ok"});
    btnGroup.add("button", undefined, "キャンセル", {name: "cancel"});
    
    if (dialog.show() == 1) {
        var year = parseInt(yearInput.text);
        var isGridLayout = layoutGrid.value;
        var calendarScale = parseFloat(sizeInput.text) / 100;
        var isJapanese = langJP.value;
        var isCenterPos = posCenter.value;
        var showGrid = gridYes.value;
        
        createCalendar(doc, year, isJapanese, isCenterPos, showGrid, artboardWidth, artboardHeight, isGridLayout, calendarScale);
    }
    
    // カレンダー作成関数
    function createCalendar(doc, year, isJapanese, isCenterPos, showGrid, artboardWidth, artboardHeight, isGridLayout, calendarScale) {
        // 既存のアートボードを削除（最後の1つは残す）
        while (doc.artboards.length > 1) {
            doc.artboards.remove(doc.artboards.length - 1);
        }
        
        // カレンダーのサイズを計算（アートボードの半分程度）
        var baseCalendarWidth = artboardWidth * 0.8; // 80%の幅を使用
        var calendarWidth = baseCalendarWidth * calendarScale; // スケールを適用
        var gridSize = calendarWidth / 7; // 7列（曜日）で割る
        
        // フォントサイズを自動計算（グリッドサイズに合わせる）
        var fontSize = gridSize * 0.3;
        var padding = gridSize * 0.1;
        
        // 左上配置の場合はグリッドサイズを調整（ただし横幅は変えない）
        var displayGridSize = gridSize;
        if (!isCenterPos) {
            // 左上配置の場合、セルを縦長にするが横幅は維持
            displayGridSize = gridSize * 1.2;
        }
        
        // カレンダーの幅は常に7セル分
        calendarWidth = gridSize * 7;
        
        // 月名と曜日名の定義
        var monthNames = isJapanese ? 
            ["1月", "2月", "3月", "4月", "5月", "6月", "7月", "8月", "9月", "10月", "11月", "12月"] :
            ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];
        
        var dayNames = isJapanese ?
            ["日", "月", "火", "水", "木", "金", "土"] :
            ["Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"];
        
        // 祝日の定義
        var holidays = getHolidays(year);
        
        // 最大のカレンダー高さを計算（6週間が最大）
        var maxCalendarHeight = fontSize * 1.5 * 1.2 + displayGridSize * 0.2 + fontSize * 0.8 * 1.2 + displayGridSize * 0.15 + displayGridSize * 6;
        var bottomMargin = displayGridSize * 0.5; // 下部余白
        
        // 各月のアートボードを作成（レイアウトに応じて配置）
        for (var month = 0; month < 12; month++) {
            var row, col;
            
            if (isGridLayout) {
                // 5-5-2の配置
                if (month < 5) {
                    row = 0;
                    col = month;
                } else if (month < 10) {
                    row = 1;
                    col = month - 5;
                } else {
                    row = 2;
                    col = month - 10;
                }
            } else {
                // 横一列の配置
                row = 0;
                col = month;
            }
            
            var artboardX = col * (artboardWidth + 100);
            var artboardY = -row * (artboardHeight + 100);
            var artboardRect = [artboardX, artboardY, artboardX + artboardWidth, artboardY - artboardHeight];
            
            var artboard;
            if (month === 0) {
                // 最初の月は既存のアートボードを更新
                artboard = doc.artboards[0];
                artboard.artboardRect = artboardRect;
                artboard.name = year + "年 " + monthNames[month];
            } else {
                // 2月目以降は新規作成
                artboard = doc.artboards.add(artboardRect);
                artboard.name = year + "年 " + monthNames[month];
            }
            
            // アートボードをアクティブにする
            doc.artboards.setActiveArtboardIndex(month);
            
            // カレンダーの配置位置（アートボードの下寄り、最大高さ基準）
            var calendarX = artboardX + (artboardWidth - calendarWidth) / 2;
            var calendarY = artboardY - (artboardHeight - maxCalendarHeight - bottomMargin); // 下から上に配置
            
            createMonthCalendar(doc, year, month, monthNames[month], dayNames, holidays, calendarX, calendarY, isCenterPos, showGrid, fontSize, gridSize, displayGridSize, padding);
        }
    }
    
    // 月のカレンダーを作成
    function createMonthCalendar(doc, year, month, monthName, dayNames, holidays, x, y, isCenterPos, showGrid, fontSize, gridSize, displayGridSize, padding) {
        var layer = doc.activeLayer;
        
        // 月名を表示
        var monthText = layer.textFrames.add();
        monthText.contents = monthName;
        monthText.textRange.size = fontSize * 1.5;
        monthText.textRange.characterAttributes.fillColor = getRGBColor(0, 0, 0);
        if (app.textFonts.length > 0) {
            monthText.textRange.characterAttributes.textFont = app.textFonts[0];
        }
        monthText.top = y;
        // 月名を中央揃え
        monthText.left = x + (gridSize * 7 - monthText.width) / 2;
        
        // 月名と曜日の間隔
        var monthSpacing = fontSize * 1.5 * 1.2 + displayGridSize * 0.2;
        
        // 曜日ヘッダーを表示
        for (var i = 0; i < 7; i++) {
            var dayText = layer.textFrames.add();
            dayText.contents = dayNames[i];
            dayText.textRange.size = fontSize * 0.8;
            
            // 日曜日と土曜日の色
            if (i === 0) {
                dayText.textRange.characterAttributes.fillColor = getRGBColor(255, 0, 0);
            } else if (i === 6) {
                dayText.textRange.characterAttributes.fillColor = getRGBColor(0, 0, 255);
            } else {
                dayText.textRange.characterAttributes.fillColor = getRGBColor(0, 0, 0);
            }
            
            dayText.top = y - monthSpacing;
            dayText.left = x + i * gridSize + (gridSize - dayText.width) / 2;
        }
        
        // カレンダーグリッドを作成
        var firstDay = new Date(year, month, 1).getDay();
        var daysInMonth = new Date(year, month + 1, 0).getDate();
        
        // 曜日とグリッドの間隔
        var headerSpacing = fontSize * 0.8 * 1.2 + displayGridSize * 0.15;
        
        var currentDay = 1;
        for (var week = 0; week < 6; week++) {
            for (var day = 0; day < 7; day++) {
                var cellX = x + day * gridSize;
                var cellY = y - monthSpacing - headerSpacing - week * displayGridSize;
                
                // グリッドを描画
                if (showGrid) {
                    var rect = layer.pathItems.rectangle(cellY, cellX, gridSize, displayGridSize);
                    rect.filled = false;
                    rect.stroked = true;
                    rect.strokeColor = getRGBColor(200, 200, 200);
                    rect.strokeWidth = 0.5;
                }
                
                // 日付を表示
                if ((week === 0 && day >= firstDay) || (week > 0 && currentDay <= daysInMonth)) {
                    if (currentDay <= daysInMonth) {
                        var dateText = layer.textFrames.add();
                        dateText.contents = currentDay.toString();
                        dateText.textRange.size = fontSize;
                        
                        // 数字の位置を調整
                        if (isCenterPos) {
                            dateText.top = cellY - (displayGridSize - dateText.height) / 2;
                            dateText.left = cellX + (gridSize - dateText.width) / 2;
                        } else {
                            dateText.top = cellY - padding;
                            
                            if (currentDay < 10) {
                                var tempText = layer.textFrames.add();
                                tempText.contents = "00";
                                tempText.textRange.size = fontSize;
                                var twoDigitWidth = tempText.width;
                                tempText.remove();
                                
                                dateText.left = cellX + padding + (twoDigitWidth - dateText.width) / 2;
                            } else {
                                dateText.left = cellX + padding;
                            }
                        }
                        
                        // 祝日チェック
                        var dateKey = year + "-" + padZero(month + 1) + "-" + padZero(currentDay);
                        var isHoliday = false;
                        for (var h = 0; h < holidays.length; h++) {
                            if (holidays[h] === dateKey) {
                                isHoliday = true;
                                break;
                            }
                        }
                        
                        // 日曜日、土曜日、祝日の色設定
                        if (isHoliday || day === 0) {
                            dateText.textRange.characterAttributes.fillColor = getRGBColor(255, 0, 0);
                        } else if (day === 6) {
                            dateText.textRange.characterAttributes.fillColor = getRGBColor(0, 0, 255);
                        }
                        
                        currentDay++;
                    }
                }
            }
            
            if (currentDay > daysInMonth) break;
        }
    }
    
    // 祝日リストを取得
    function getHolidays(year) {
        var holidays = [];
        
        holidays.push(year + "-01-01"); // 元日
        holidays.push(year + "-02-11"); // 建国記念の日
        holidays.push(year + "-02-23"); // 天皇誕生日
        holidays.push(year + "-04-29"); // 昭和の日
        holidays.push(year + "-05-03"); // 憲法記念日
        holidays.push(year + "-05-04"); // みどりの日
        holidays.push(year + "-05-05"); // こどもの日
        holidays.push(year + "-08-11"); // 山の日
        holidays.push(year + "-11-03"); // 文化の日
        holidays.push(year + "-11-23"); // 勤労感謝の日
        
        holidays.push(getNthWeekday(year, 0, 1, 2)); // 成人の日
        holidays.push(getNthWeekday(year, 6, 1, 3)); // 海の日
        holidays.push(getNthWeekday(year, 8, 1, 3)); // 敬老の日
        holidays.push(getNthWeekday(year, 9, 1, 2)); // スポーツの日
        
        holidays.push(year + "-03-20"); // 春分の日（近似）
        holidays.push(year + "-09-23"); // 秋分の日（近似）
        
        return holidays;
    }
    
    // N番目の特定曜日を取得
    function getNthWeekday(year, month, weekday, n) {
        var date = new Date(year, month, 1);
        var count = 0;
        
        while (date.getMonth() === month) {
            if (date.getDay() === weekday) {
                count++;
                if (count === n) {
                    return year + "-" + padZero(month + 1) + "-" + padZero(date.getDate());
                }
            }
            date.setDate(date.getDate() + 1);
        }
        return "";
    }
    
    // ゼロパディング
    function padZero(num) {
        return (num < 10 ? "0" : "") + num;
    }
    
    // RGB色を作成
    function getRGBColor(r, g, b) {
        var color = new RGBColor();
        color.red = r;
        color.green = g;
        color.blue = b;
        return color;
    }
})();